;;; 
;;; Virtual Disk Drive Project	- Sep 8, 2001 - EJR (in Austin)
;;;
;;;

;;; Serial - interface to the UART

#include "vd.h"
#include "disk.h"
#include "utility.h"
#include "track.h"

;;;
;;; This routine exports the following symbols
;;; 
	global SerialSetup         ; sets up serial port & functions
	global SerialDispatch      ; called upon interrupt by the serial port
	
;;; Formula from documentation:
;;; BRGH = 1        BAUD = Fosc/(16(SPBRG+1))
;;; BRGH = 0        BAUD = Fosc/(64(SPBRG+1))
;;;
;;; lisp code to do the math:

#ifdef NOTDEF

   THIS ROUTINE WILL CALCULATE RESULTANT BAUD AND "CLOSENESS"

   (let ((Fosc   20000000)
	 (BRGH   0)
	 (SPBRG  2)
	 (Target 115200))
     (let ((BAUD (/ Fosc (* (if (= BRGH 0) 64 16) (+ SPBRG 1)))))
	(concat "Baud: " (int-to-string BAUD)
		" (" (int-to-string Target) " "
		     (number-to-string (/ BAUD (* Target 1.00))) ")")))


    THIS ROUTINE WILL SOLVE FOR A TARGET
	
    (let ((Fosc 20000000)
	  (Target 115200))
      (let ((brgh1 (- (/ Fosc 16.0 Target) 1))
	    (brgh0 (- (/ Fosc 64.0 Target) 1)))
	(let ((int1 (truncate (+ brgh1 .5)))
	      (int0 (truncate (+ brgh0 .5)))
	      (approx1 (/ (truncate (* brgh1 1000)) 1000.0))
	      (approx0 (/ (truncate (* brgh0 1000)) 1000.0)))
	  (let ((baud1 (ftruncate (/ Fosc (* 16.0 (+ int1 1)))))
	        (baud0 (ftruncate (/ Fosc (* 64.0 (+ int0 1))))))
	    (let ((var1 (/ (ftruncate (* (- (/ Target baud1) 1) 10000)) 100))
	          (var0 (/ (ftruncate (* (- (/ Target baud0) 1) 10000)) 100)))
	(concat "(BRGH(1)--> " (int-to-string int1) " "
				      (int-to-string approx1) " "
				      (int-to-string baud1) " "
				      (int-to-string var1) "%) "
	        "(BRGH(0)--> " (int-to-string int0) " "
				      (int-to-string approx0) " "
				      (int-to-string baud0) " "
				      (int-to-string var0) "%) "))))))

#endif

	
#define BAUD_BYTE	d'21'	; 57600 at 20MHz (BRGH 1)
#define BAUD_CLOCK	1

;;#define BAUD_BYTE	d'10'	; 115200 at 20MHz (BRGH 1)
;;#define BAUD_CLOCK	1

;;#define BAUD_BYTE	d'64'	; 19200 at 20MHz (BRGH 1)
;;#define BAUD_CLOCK	1

PROG1 CODE

;;;**********************************************************************
;;; NAME:	StaticBaud			()
;;;
;;; DESCR:	Sets the static baud rate as defined above
;;;
;;; ARGS:	
;;;
;;; RETURNS:	
;;;
;;; NOTES:	
;;;**********************************************************************
StaticBaud:
	BankSelect 1		; switch to bank one for these settings

	movlw	BAUD_BYTE	; set baud rate (defined above)
	movwf	SPBRG

#if BAUD_CLOCK == 1
	bsf	TXSTA, BRGH	; high baud rate (1)
#else
	bcf	TXSTA, BRGH	; low baud rate (0)
#endif

	BankSelect 0		; and back to bank zero to go on

	return
	
;;;**********************************************************************
;;; NAME:	AutoBaud()
;;;
;;; DESCR:	Autobauds the serial line.  Normally called upon reset.
;;;		Requires the first caracter to be ^A or something with
;;;		all 0's followed by all 1's.
;;;
;;; ARGS:	
;;;
;;; RETURNS:	
;;;
;;; NOTES:	___     _______________       ________
;;;                \___/               \_____/
;;;
;;;                    |--------- p ---------|
;;;                |-s-|------- data --------|
;;;
;;;     s = start bit
;;;  data = serial data (ignorning any stop bits)
;;;     p = the number of instructions from the end of the stop bit to end of data
;;;
;;; This routine counts the number of instructions (p), modified by the 1:16
;;; pre-scaler to arrive at a BAUD count to be set in SPBRG.
;;;     SPBRG = (p / 32) - 1     BRGH = 1
;;; (note that the prescaler is 1:16 and a divide by 2 is done later)
;;;**********************************************************************
#define RX	7
AutoBaud:
	BankSelect	1

	bcf	OPTION_REG,T0CS	; set source of TMR0 to internal instr clock
	bcf	OPTION_REG,PSA	; set prescaler assignment to TMR0
	bcf	OPTION_REG,PS2	; write 011 to prescaler = 1:16
	bsf	OPTION_REG,PS1	; (NOTE - since TMR0 isn't used anywhere else, and
	bsf	OPTION_REG,PS0	;  neither is WDT, these settings aren't ever changed)
	

	BankSelect	0

	bcf	RCSTA, CREN	; turn off serial receive processing
	call	WaitForRise

	clrf	TMR0
	call	WaitForRise

	movf	TMR0,W		; got the count 
	movwf	WORK0		; save it
	bcf	STATUS,C	; clear carry preparing for rotate
	rrf	WORK0,W		; divide by 2 (rotate)
	btfss	STATUS,C	; if carry, then -1 is already done
	addlw	0xff		; otherwise, do the -1

	BankSelect	1
	movwf	SPBRG		; set the baud rate
	bsf	TXSTA, BRGH	; must be in high-speed mode for this to work
	BankSelect	0	; 

	return

WaitForRise:
	btfsc	PORTC,RX	; wait until the line goes low on start bit or data
	goto	WaitForRise

;;; 	clrwdt			; this was here in the original source for autobauding
	
WaitForRise2:
	btfss	PORTC,RX	; now wait for the line to go high at end of start bit or data
	goto	WaitForRise2

	return

		
;;;*******************************************************************
;;; NAME:	SerialSetup()
;;;
;;; DESCR:	Sets up the serial system including state.
;;;
;;; ARGS:	
;;;
;;; RETURNS:	
;;;
;;; NOTES:	
;;;*******************************************************************

SerialSetup:

	BankSelect	1
	clrf	TXSTA
	bsf	TXSTA, CSRC	; clock from BRG (1) (really no-op for async)

	bsf	TXSTA, TXEN	; enable transmission (1)
	bcf	TXSTA, SYNC	; async mode (0)
	BankSelect	0

;;; 	call	StaticBaud	; statically set the baud rate from above
	call	AutoBaud	; automagically set the baud on an ^A
	
	clrf	RCSTA
	bsf	RCSTA, SPEN	; serial port enabled (1)
	bsf	RCSTA, CREN	; continuous receive (1)
	movf	RCREG,W		; clear any whacky incoming data
	movf	RCREG,W		; (clear the 2 level buffer)

	LOAD_STATE_0		; return load state to zero

	call	ResetCmd	; ensure all is ready to load
	call	SendVersion	; send out version (indicates serial trained)
	call	XmitWait	; send out a ^A after the version
	movlw	0x01		; (this mirrors what happens in the
	movwf	TXREG		;   command loop)
	
	return
	
;;;*******************************************************************
;;; NAME:	SendVersion()
;;;
;;; DESCR:	Send out the version number to the serial port.
;;;		Assumes that the serial port is already set up.
;;;
;;; ARGS:	
;;;
;;; RETURNS:	
;;;
;;; NOTES:	
;;;*******************************************************************
SendVersion:
	PrintByte	'S'	; "SVD 2.3"
	PrintByte	'V'
	PrintByte	'D'
	PrintByte	' '
	PrintByte	'2'
	PrintByte	'.'
	PrintByte	'3'

	return

;;;*******************************************************************
;;; NAME:	SerialDispatch()
;;;
;;; DESCR:	This routine dispatches a serial byte to the appropriate
;;;		routine based upon the current state.  Called from the
;;;		interrupt routine.
;;;
;;; ARGS:	W holds the byte taken off of the serial port
;;;
;;; RETURNS:	This routine assumes an interrupt return from
;;;		the routines it calls.
;;;
;;; NOTES:	- this routine could use a table, but it will be easier
;;;		  to simply use "ifs"
;;;		- it uses a "high-hot" approach where states are always
;;;		  incrementing...that is, to move to the next state,
;;;		  just set the next higher bit...lower bits are ignored.
;;;*******************************************************************

SerialDispatch:

	btfsc	LOAD_STATE_8	; state '1XXX XXXX'
	goto	StateDebug

	btfsc	LOAD_STATE_7	; state '01XX XXXX'
	goto	DiskDump	; dump the disk (tied to state 6)

	btfsc	LOAD_STATE_6	; state '001X XXXX' - start of dump disk 
	goto	DiskDumpParams	; dump the disk parameters first

	btfsc	LOAD_STATE_5	; state '0001 XXXX'
	goto	StateLoadData

	btfsc	LOAD_STATE_4	; state '0000 1XXX'
	goto	StateLoadSecSize
	
	btfsc	LOAD_STATE_3	; state '0000 01XX'
	goto	StateLoadTracks

	btfsc	LOAD_STATE_2	; state '0000 001X'
	goto	StateLoadSecs

	btfsc	LOAD_STATE_1	; state '0000 0001'
	goto	StateLoadDisk

	;; LOAD_STATE_0
	goto	Command	; state '0000 0000'

;;;*******************************************************************
;;; Serial Commands
;;;
;;; Here are the commands that are handled serially.  All echo back 
;;; the command after running:
;;;
;;;			"^@" - 0x00 = nop - just returns the same thing
;;;			"^A" - 0x01 = print version number
;;;		        "^B" - 0x02 = dump a disk, next byte is disk number
;;;			"^D" - 0x04 = reset (does a hard reset)
;;;		        "^H" - 0x08 = start the disk(s) running
;;;			"^P" - 0x10 = stop the disk(s) running
;;;			" "  - 0x20 = load a disk, next bytes are: disk#, sects, tracks, secsize, then data
;;;			"@"  - 0x40 = extended debug commands (see below)
;;;		     "alt-@" - 0x80 = unused
;;;		
;;;		Debug Commands:
;;;			"@^@" - nop
;;;			"@^A" - move head out (increase)
;;;			"@^B" - move head in (decrease)
;;;			"@^D" - move to track zero
;;;			"@^H" - dump state of disk 0
;;;			"@^P" - dump state of disk 1
;;;			"@ "  - dump state of disk 2
;;;			"@@"  - nop
;;;			"@alt-@"  - nop
;;;
;;;*******************************************************************

;;;*******************************************************************
;;; NAME:	Command()
;;;
;;; DESCR:	State 0 - idle serial state.  Commands are interpreted
;;;		at this state.
;;;
;;; ARGS:	Assumes input byte in W.
;;;		Uses SER0
;;;
;;; RETURNS:	
;;;
;;; NOTES:	- command is given as "high-hot", commands:
;;;    "^@" - 0x00	0000 0000 --> (nop - always)
;;;    "^A" - 0x01	0000 0001 --> "Print Version number"
;;;    "^B" - 0x02	0000 001X --> "Dump" (state 6)
;;;    "^D" - 0x04	0000 01XX --> "Reset" (no states - implies Stop)
;;;    "^H" - 0x08	0000 1XXX --> "Run" (no states)
;;;    "^P" - 0x10	0001 XXXX --> "Stop" (no states)
;;;    " "  - 0x20	001X XXXX --> "Load" (states 1, 2, 3,4)
;;;    "@"  - 0x40	01XX XXXX --> "Debug" (state 7)
;;; "alt-@" - 0x80	1XXX XXXX --> (nop - reserved for later)
;;;		- undefined commands will just return
;;;		- all commands cause a "feedback byte" to be returned
;;;		  which is often simply the command returned
;;;		- this can be used to get to a known state, just keep
;;;		    sending 0's until they start coming back.
;;;*******************************************************************
Command:
	movwf	SER0
	
	MemReadMode		; command state will always set read mode
	
	btfsc	SER0,0		; command '0000 0001' (^A)
	call	SendVersion	;   sends out version (with trailing ^A of course below)

	btfsc	SER0,1		; command '0000 0010' (^B)
	bsf	LOAD_STATE_6	;   next byte is disk to dump

	btfsc	SER0,2		; command '0000 0100' (^D)
	goto	0x0000		; hard core reset for now - don't worry about stack
;	call	ResetCmd

	btfsc	SER0,3		; command '0000 1000' (^H)
	clrf	RUNSTATE	; get the thing running!

	btfsc	SER0,4		; command '0001 0000' (^P)
	bsf	RUNSTATE,STOPPING  ; get the stopping started

	btfsc	SER0,5		; command '0010 0000' (" ")
	bsf	LOAD_STATE_1	;   next byte begins loading a disk
	
	btfsc	SER0,6		; command '0100 0000' ("@")
	bsf	LOAD_STATE_8	;   next byte is debug command

	btfsc	SER0,7		; command '1000 0000' (alt-@)
	nop			;   nop so far
	
				; command '0000 0000' falls through
	call	XmitWait
	movf	SER0,W
	movwf	TXREG		; send out ack byte

	return
	
;;;*******************************************************************
;;; NAME:	ResetCmd()
;;;
;;; DESCR:	Resets the disk storage, effectively wiping out all
;;;		data and such.
;;;
;;; ARGS:	Uses ARG0 & ARG1 'cause of DiskSetCopy
;;;
;;; RETURNS:	
;;;
;;; NOTES:	- No change to LOAD_STATE
;;;		- since memory is clear, it doesn't really matter where
;;;		  the low address pointer is
;;;*******************************************************************
ResetCmd:
	bsf	RUNSTATE,STOPPING
	bsf	RUNSTATE,STOPPED   ; indicate stopped if not otherwise

	clrf	TOP_LO		; clear top memory pointer
	clrf	TOP_HI

	movlw	NULLDISK
	movwf	CURDISK		; invalidate current disk

	movlw	ACTIVEDISK
	movwf	ARG0
	movlw	DISK0
	movwf	ARG1
	call	DiskSetCopy	; invalidate disk 0

	movlw	ACTIVEDISK
	movwf	ARG0
	movlw	DISK1
	movwf	ARG1
	call	DiskSetCopy	; invalidate disk 1
	
	movlw	ACTIVEDISK
	movwf	ARG0
	movlw	DISK2
	movwf	ARG1
	call	DiskSetCopy	; invalidate disk 2
	
	return

;;;*******************************************************************
;;; NAME:	MapDiskNumber()
;;;
;;; DESCR:	Maps a disk number to a disk pointer.  Also notices
;;;		if the special bit (bit 2) is set indicating a WP disk.
;;;		Can do 4 disks (0 thru 3).
;;;
;;;		REQDISK is set to the right disk pointer
;;;		WRITEPROT is zero if no write protect was specified
;;;
;;; ARGS:	W is the disk NUMBER
;;;
;;; RETURNS:	REQDISK has the current disk pointer to work with
;;;
;;; NOTES:	WORK0 is used for temporary storage
;;;*******************************************************************
MapDiskNumber:

	movwf	SER1
	clrf	WRITEPROT	; planning ahead
	
	btfsc	SER1,2		; check for special bit
	incf	WRITEPROT	; set protect flag if it is set

	bcf	SER1,2		; in any case, clear that bit!

	;; now map the disk number to a pointer

	movf	SER1		; just sets the flags
	skpnz
	movlw	DISK0

	decf	SER1
	skpnz
	movlw	DISK1

	decf	SER1
	skpnz
	movlw	DISK2

	decf	SER1
	skpnz
	movlw	DISK3

	movwf	REQDISK		; move disk pointer to cur disk

	return

;;;*******************************************************************
;;; NAME:	StateDebug()
;;;
;;; DESCR:	Incoming byte is a debug command. Commands:
;;;			no bits -->
;;;			bit 0   --> head out (increase)
;;;			bit 1   --> head in (decrease)
;;;			bit 2   --> move to track zero
;;;			bit 3   --> dump disk #0 state
;;;			bit 4   --> dump disk #1 state
;;;			bit 5   --> dump disk #2 state
;;;			bit 6   -->
;;;			bit 7   -->
;;;		Combine bits if you want to run multiple commands.
;;;		Commands are run from bits 0 to 7 (right to left).
;;;
;;; ARGS:	W has the incoming byte
;;;
;;; RETURNS:	nada
;;;
;;; NOTES:	SER2 is used
;;;*******************************************************************
StateDebug:

	movwf	SER2
	skpnz
	nop			; command '0000 0000'

	btfsc	SER2,0	
	call	NextTrack	; command '0000 0001'

	btfsc	SER2,1	
	call	PrevTrack	; command '0000 0010'

	btfsc	SER2,2	
	call	ZeroTrack	; command '0000 0100'

	btfsc	SER2,3
	call	DiskState0	; command '0000 1000'	

	btfsc	SER2,4	
	call	DiskState1	; command '0001 0000'

	btfsc	SER2,5	
	call	DiskState2	; command '0010 0000'
	
	btfsc	SER2,6	
	nop			; command '0100 0000'

	btfsc	SER2,7	
	nop			; command '1000 0000'
	
	LOAD_STATE_0		; goto command state

	return

;;;*******************************************************************
;;; NAME:	StateLoadDisk(), ...Tracks(), ...Secs() ...SecSize()
;;;
;;; DESCR:	StateLoadDisk() - Loads the disk number to work with.
;;;		StateLoadTracks() - Loads the number of tracks 
;;;		StateLoadSecs()   - Loads the number of sectors
;;;		StateLoadSecSize()- Loads the sector size
;;; 
;;; ARGS:	W has the incoming byte
;;;
;;; RETURNS:	
;;;
;;; NOTES:  - StateLoadSecSize() is meant to be the last call before
;;;		loading sector data, so it sets up the data necessary
;;;		for appropriate loading of the data.
;;;	    - Sector sizes are interpreted as follows:
;;;			0 = 128 bytes
;;;			1 = 256
;;;			2 = 384
;;;			3 = 512
;;;			...
;;;			7 = 1024
;;;	    - StateLoadDisk() is used for both loading and dumping.
;;;*******************************************************************
StateLoadDisk:
	call		MapDiskNumber
	LOAD_STATE_NEXT		; move to next state...which can be anything
	return

StateLoadTracks:
	movwf		TRACKS
	LOAD_STATE_NEXT		; prepare for next state
	return

StateLoadSecs:
	movwf		SECTORS
	LOAD_STATE_NEXT		; prepare for next state
	return

StateLoadSecSize:	
	movwf		SECSIZE
	LOAD_STATE_NEXT			; prepare for next state
	call		PrepareForData	; get ready for incoming data
	return

;;;*******************************************************************
;;; NAME:	BaseAlign()
;;;
;;; DESCR:	Align the base for the incoming disk according to the
;;;		number of incoming sectors.  Also, calculate the info
;;;		(REMAINDER) that allows easy track movement.
;;;
;;; ARGS:	W is the number of blocks for a track (normally this is
;;;		the number of sectors + 1)
;;;
;;; RETURNS:	BASE_HI/LO is reset to start at the right spot
;;;		REMAINDER is set right
;;;
;;; NOTES:
;;; 
;;;   The idea is to make sure that accessing a track (made up of a
;;; number of sectors) never crosses the boundary of a block of pointers.
;;; In this case, 8 bits.  So, after getting the number of sectors (+1
;;; for the header info of course), figure out two things:
;;;   - the REMAINDER - this is the number of blocks that are skipped
;;;     before the 8 bits rolls over.  By skipping this remainder, we
;;;     can be assured that we don't roll over accidentally.  More
;;;     importantly, by knowing this remainder, when counting tracks
;;;     down, we know how much to subtract after rolling under 0.
;;;   - the BASE - to make things easy, the first track will always
;;; 	start on either 0 or a multiple of SECTORS.  In this way,
;;;	even if a disk spans 256 blocks, we don't have to worry about
;;;     keeping multiple remainders (in the off case that the track
;;;     starts just before the end of the first segment, and spans
;;;     the second segment, rolling into the third).
;;;   - to calculate the REMAINDER - get the number of sectors and
;;;     divide it into 256, get the remainder, simple.
;;;   - to calculate the BASE - divide the current base by sectors,
;;;     and add the remainder to the current base to set the new base
;;; In both calculations, special case the zero remainder case if
;;; necessary.
;;;*******************************************************************
BaseAlign:
	movf	TRACKBLOCKS,W	
	call	Mod256
	movwf	REMAINDER

	;; simple base calculation

	movf	BASE_LO, W
	movwf	ARG0
	movf	TRACKBLOCKS, W
	call	ModInverse

	addwf	BASE_LO
	skpc
	return

	;; we carried here, need to reset LO and incr HI

	clrf	BASE_LO		; reset to zero
	incf	BASE_HI

	return

;;;*******************************************************************
;;; NAME:	PrepareForData()
;;;
;;; DESCR:	Given that all of the important data has been loaded,
;;;		calculate things necessary for data load.
;;;
;;; ARGS:	
;;;
;;; RETURNS:	
;;;
;;; NOTES:	- uses ARG0 to call Multiply()
;;;*******************************************************************
PrepareForData:
	;; given the sector size, calculate the track blocks, for example:	
	;; if 0 (=128) then trackblocks = (sectors+1)/2
	;; if 1 (=256) then trackblocks = sectors
	;; if 2 (=384) then trackblocks = sectors + ((sectors+1)/2)
	;; if 3 (=512) then trackblocks = sectors*2
	;; if 4 (=640) then trackblocks = sectors*2 + ((sectors+1)/2)
	;; if 5 (=768) then trackblocks = sectors*3
	;; if 6 (=896) then trackblocks = sectors*3 + ((sectors+1)/2)
	;; if 7 (=1024) then trackblocks = sectors*4
	;; generic formula: 
	;;   (sectors * (secsize+1)/2)
	;;   plus  ((sectors+1)/2)  if multiple of 128

	clrf	ARG0		; compute (secsize+1)/2
	incf	ARG0
	movf	SECSIZE,W
	addwf	ARG0		; clears carry too
	rrf	ARG0		; ARG0 now has (secsize+1)/2

	;; compute (sectors * (secsize+1)/2)

	movf	SECTORS,W
	call	Multiply	; W now has (sectors * (secsize+1)/2)
	movwf	TRACKBLOCKS

	;; compute (sectors+1)/2    (in case we need it)
	
	clrf	ARG0
	incf	ARG0
	movf	SECTORS,W
	addwf	ARG0		; clears carry too
	rrf	ARG0,W		; W now has (sectors+1)/2

	;; if we have a 128-byte multiple, add in (sectors+1)/2

	btfss	SECSIZE,0
	addwf	TRACKBLOCKS	; only add if bit 0 is clear

	incf	TRACKBLOCKS	; add one for the header block
	
	;; get block memory pointers set up

	movf	TOP_LO,W
	movwf	BASE_LO		; sets disk LO
	movf	TOP_HI,W
	movwf	BASE_HI		; sets disk HI

	;; may have to realign block memory pointers to ensure
	;; that block roll-over upon track seek works ok..

	call	BaseAlign

	;; prepare track counter for data load
	
	movf	TRACKS,W
	movwf	REMTRACKS	; count down of tracks

	;; increment sector count and prepare sector counter for data load
	
	movf	TRACKBLOCKS,W
	movwf	REMBLOCKS	; sectors count down during load

	;; put the requested disk into CURDISK
	
	movf	REQDISK,W
	movwf	CURDISK
	
	;; prepare for loading data at this point

	call	ZeroTrack

	MemAddrSet		; set the current block mem address

	MemWriteMode		; set write mode on RAM

	clrf	SER3		; prepare data count

	return
	
;;;*******************************************************************
;;; NAME:	StateLoadData
;;;
;;; DESCR:	Reads the incoming data loading it into memory
;;;
;;; INPUTS:	W holds serial byte
;;;
;;; RETURNS:	
;;; 
;;; NOTES:	- This routine is blissfully stupid about blocks.
;;;		  it just loads them.  So the data has to be pre-
;;;		  composed correctly.  This means that even the header
;;;		  block loads a whole block.
;;;		- this routine could be coded to take header data
;;;		  followed by the the sector data...but what the
;;;		  heck, let the PC do some work.
;;; NOTES:	- SER3 used to keep a count of incoming bytes - it is
;;;		  allowed to just roll-over and roll-over
;;; *******************************************************************
StateLoadData:

	MemLoadByte		; load W to memory write register
 	MemWriteByte		; output current byte to mem
 	MemLowInc		; increment low mem pointer
;; 	MemWriteLowInc		; testing, testing...

	decfsz	SER3		; 256 bytes come in yet?
	return			; not done with block/sector

	;; now move to next sector, it is save to just increment
	;; the mem pointer because we WON'T rollover on it

	incf	MEMBLOCKLO	; direct increment of address lines

	decf	REMBLOCKS	; count down on number of blocks to load
	skpz			; done with a track yet?
	return			; not done with track

	;; at this point, we move to the next track
	
	movf	TRACKBLOCKS,W
	movwf	REMBLOCKS	; sectors reset for sector countdown

	call	NextTrack	; prepare for next track

	decf	REMTRACKS	; count down number of tracks
	skpz			; done with all tracks?
	return			; not done with tracks
	
				; done with all data load

	;; update the TOP pointers
	;;   note that CURBASE_LO is set to the start of the last track
	;;   so incrementing it by TRACKBLOCKS will roll-over ONLY if it was
	;;   a perfect fit.

	movf	CURBASE_HI,W
	movwf	TOP_HI
	movf	CURBASE_LO,W
	movwf	TOP_LO
	
	movf	TRACKBLOCKS,W
	addwf	TOP_LO
	skpnz
	incf	TOP_HI

	call	ZeroTrack	; go back to the beginning of the disk

	movf	CURDISK,W
	call	DiskActivate	; checkpoint this disk
	
	LOAD_STATE_0		; goto command state
	return

;;;*******************************************************************
;;; NAME:	DiskState#() and DiskStateDump()
;;;
;;; DESCR:	Dumps the state of the given disk.
;;;		This thing just dumps it out in one
;;;		fell swoop.  Since it is within the context of an
;;;		interrupt, nothing can interrupt this process.
;;;
;;; ARGS:	
;;;
;;; RETURNS:	
;;;
;;; NOTES:	- W is set to return the last byte which is done
;;;		  by the caller.
;;;*******************************************************************
DiskState0:
	movlw	DISK0
	goto	DiskStateDump
	
DiskState1:	
	movlw	DISK1
	goto	DiskStateDump
	
DiskState2:	
	movlw	DISK2
	goto	DiskStateDump
	
DiskStateDump:

	movwf	REQDISK

	movf	CURDISK,W
	call	DiskActivate	; checkpoint current disk

	movf	REQDISK,W
	call	DiskActivate	; load the requested disk
	
	movf	BASE_HI,W
	movwf	ARG0
	movf	BASE_LO,W
	movwf	ARG1
	movlw	'B'
	call	SerPrint2

	movf	TRACKS,W
	movwf	ARG0
	movlw	'T'
	call	SerPrint1

	movf	SECTORS,W
	movwf	ARG0
	movlw	'S'
	call	SerPrint1

	movf	SECSIZE,W
	movwf	ARG0
	movlw	'#'
	call	SerPrint1

	movf	TRACKBLOCKS,W
	movwf	ARG0
	movlw	'b'
	call	SerPrint1

	movf	CURTRACK,W
	movwf	ARG0
	movlw	'='
	call	SerPrint1

	movf	CURBASE_HI,W
	movwf	ARG0
	movf	CURBASE_LO,W
	movwf	ARG1
	movlw	'C'
	call	SerPrint2

	movf	REMAINDER,W
	movwf	ARG0
	movlw	'R'
	call	SerPrint1

	;; BIG NOTE - previous disk not currently restored!
	;; restore the previous disk
	
	return

;	movf	TOP_HI,W
;	movwf	ARG0
;	movf	TOP_LO,W
;	movwf	ARG1
;	movlw	'F'
;	call	SerPrint2

;;;*******************************************************************
;;; NAME:	DiskDumpParams()
;;;
;;; DESCR:	Dumps the parameters of the requested disk.  These
;;;		are the sectors, tracks, secsize, and writeprot.
;;;
;;; ARGS:	
;;;
;;; RETURNS:	
;;;
;;; NOTES:	
;;;*******************************************************************
DiskDumpParams:
	call	StateLoadDisk	; get the disk number requested
				; (sets up for next state too) 
	MemReadMode
	
	movf	CURDISK,W
	call	DiskActivate	; checkpoint current disk
	
	movf	REQDISK,W
	call	DiskActivate	; load the requested disk

	movlw	NULLDISK
	subwf	CURDISK,W
	skpnz
	goto	DiskDumpNot	; disk is not loaded
	
	;; dump the sector count, tracks, sector size

	movf	SECTORS,W	; number of sectors
	call	XmitWait
	movwf	TXREG

	movf	TRACKS,W	; number of tracks
	call	XmitWait
	movwf	TXREG

	movf	SECSIZE,W	; number of blocks per sector
	call	XmitWait
	movwf	TXREG

	movf	WRITEPROT,W	; write protect status
	call	XmitWait
	movwf	TXREG

	call	ZeroTrack	; to beginning of disk

	movf	TRACKS,W	; tracks to dump
	movwf	REMTRACKS

	movf	TRACKBLOCKS,W	; blocks in each track
	movwf	REMBLOCKS	; (sectors * secsize + 1)

	movlw	'+'		; indicator of "ready to xmit"
	call	XmitWait
	movwf	TXREG
	
	return			; ready to start dumping

DiskDumpNot:	
	movlw	'X'		; indicator of "no disk loaded here"
	call	XmitWait
	movwf	TXREG

	LOAD_STATE_0		; back to state 0
	
	return
	
;;;*******************************************************************
;;; NAME:	DiskDump()
;;;
;;; DESCR:	Dumps the next block of the disk.  Note that blocks
;;;		are 256 bytes and may be smaller/bigger than an actual
;;;		sector.
;;;
;;;		After each block, the "read for more" indicator "+"
;;;		is sent back.  After each track, the NextTrack routine
;;;		sends out a ">".  After the last track, NextTrack is
;;;		NOT called and an "end of disk" indication "!" is sent
;;;		back.
;;;
;;;		The host must send a "directive" to start the next block
;;;		dumping.  The directives are as follows:
;;;			S = stop dumping!
;;;			P = retransmit last block (may be tough to do)
;;;			N = transmit next block 
;;;		
;;; ARGS:	W has the directive byte
;;; 
;;;
;;; RETURNS:	
;;;
;;; NOTES:	- SER3 is used
;;;*******************************************************************
DiskDump:
	movwf	SER3

	;; by using ^A to do a stop, we can get back in control with
	;; a string of them.

	movlw	1		; check for "stop" directive (^A)
	subwf	SER3,W
	skpnz
	goto	DiskDumpEnd

	movlw	'P'		; check for "previous track" directive
	subwf	SER3,W
	skpnz
	goto	DiskDumpPrevTrack
	
	movlw	'N'		; check for "next track" directive
	subwf	SER3,W
	skpnz
	goto	DiskDumpNextTrack

	movlw	'p'		; check for "previous block" directive
	subwf	SER3,W
	skpnz
	goto	DiskDumpPrevBlock
	
	movlw	'n'		; check for "next block" directive
	subwf	SER3,W
	skpnz
	goto	DiskDumpNextBlock

	;; default is to stop

	goto	DiskDumpEnd


;;;*******************************************************************
;;; NAME:	DiskDumpBlock()
;;;
;;; DESCR:	Dumps out a block of data.
;;;
;;; ARGS:	SER3 counts the bytes
;;;		SER2 maintains a simple h17-style checksum
;;;
;;; RETURNS:	
;;;
;;; NOTES:	- 257 bytes are sent, 256 data and 1 checksum
;;;*******************************************************************
DiskDumpBlock:	;; called by many routines in this group to send block of data

	clrf	SER3		; used to track the byte count in a block
	clrf	SER2		; clear the checksum

DDBLoop:	
	MemReadByte		; get byte from memory
	call	XmitWait	;   and wait till transmit clear
	movwf	TXREG		;   then transmit it!

	;; build-up the checksum byte with a simple h18-style checksum xor/rotate
	;; (note that we need to take the CARRY and move it in bit 0)

	xorwf	SER2		; xor the byte into the checksum register
	rlf	SER2		; and shift for the next byte
	bcf	SER2,0		; prepare bit 0 (set from previous CARRY contents)
	btfsc	STATUS,C
	bsf	SER2,0		; if the CARRY is set, then set (SER2,0)
	
	MemLowInc		; incr low mem ptr

	decfsz	SER3		; loop until we've done 256 bytes
	goto	DDBLoop		;    (one block)

	;; send out the 257'th byte, the checksum (each block has a checksum)

	movf	SER2,W
	call	XmitWait
	movwf	TXREG
	
	incf	MEMBLOCKLO	; direct increment of address lines
	
	return


	
DiskDumpNextTrack:	;; dump the entire next track
			;; NOTE - if dumping block by block THEN doing this,
			;;   the blocks REMAINING in the track are dumped

	call	DiskDumpBlock
	
	;; done with a block, prepare for the next one

	decfsz	REMBLOCKS
	goto	DiskDumpNextTrack		; not done with the track yet

	;; done with the track here, prepare for next track

	goto	DiskDumpNextPrepare
		
DiskDumpNextBlock:	

	call	DiskDumpBlock

	decfsz	REMBLOCKS
	goto	DiskDumpReturn	; not done with the track yet
	
	;; done with the track, fall through

DiskDumpNextPrepare:	

	;; done with a track, so first reset the blocks for the next track
	
	movf	TRACKBLOCKS,W	; blocks in each track
	movwf	REMBLOCKS	; (sectors * secsize + 1)

	;; note that the NextTrack indicator will be sent out no
	;; matter what...even if we try to advance past the last
	;; track...this means that an end-of-track indicator will
	;; come out even on the last track.
	
	call	XmitWait	; need to ensure that data is gone before
	call	NextTrack	;  the NextTrack-generated marker is sent
	
	decfsz	REMTRACKS
	goto	DiskDumpReturn

	;; otherwise, we're done, so send out end of dump
	
DiskDumpEnd:		

	movlw	'!'		; indicator of "done with this disk"
	call	XmitWait	; (NOTE - this can be used by the "stop" directive
	movwf	TXREG		;   in which case the indicator is used to acknowledge the stop

	call	ZeroTrack	; to beginning of disk	

	LOAD_STATE_0		; back to state 0
	
	return

DiskDumpReturn:
	movlw	'+'		; indicator of "ready to xmit next thing"
	call	XmitWait
	movwf	TXREG

	return

;; NOTE NOTE NOTE NOTE NOTE
;; the following routines HAVE not been completed...there are here
;;   because I didn't want to lose the code...but they aren't used

DiskDumpPrevBlock:
	;; called to dump the previous block...this is nasty if we
	;; did a track move on the previous block

	;; Note that this routine is not good at trying to do a "previous"
	;; on the FIRST block...so DON'T DO IT!

	movf	TRACKBLOCKS,W
	subwf	REMBLOCKS,W
	skpz			; we just did a NextTrack, so back-up
	goto	DiskDumpPSimple

DiskDumpPrevTrack:	
	
	clrf	REMBLOCKS	; set-up for back-up
	
	;; at this point, we're on the wrong track, so we need to reverse
	;; to the previous track, then add in the blocks

	;; remember that PrevTrack generates the movement marker, so the
	;; caller should be aware that this marker will be BEFORE the data.

	call	XmitWait	; need to ensure that data is gone before
	call	PrevTrack	;  the PrevTrack-generated marker is sent
	
	MemAddrInc	TRACKBLOCKS	; move just past last block

	;; fall through, decrementing to last block of prev track

DiskDumpPSimple:

	;; in either case (a simple dump or a track back-up) we increment
	;; the number of blocks remaining to be dumped
	
	incf	REMBLOCKS	; when backing up, we have one left in track
	
	;; just a simple previous, back up the pointers and start dumping

	decf	MEMBLOCKLO	; direct DECREMENT of address lines
	goto	DiskDumpNextBlock
	
;;; =======================================================================
;;; =======================================================================

	END

